'
' Macro to import all .csv files with indicator mark in file name
' Main entry for makro: ImportIndicatorFiles
' Tested with Excel 6.5 from 2006 from PT 19.08.2010
'


Option Explicit


Private Type Config
  DefaultFileExtension As String
  DefaultFolderName As String
  FileNamePattern As String
  IndicatorMark As String
  IndicatorRegExpPattern As String
  MaxDuplicateExtension As String
  MaxOfAllowedSheets As Integer
  MaxSheetNameLength As Integer
End Type

Private Config As Config


Sub InitConfig()
'
' set the configuration
'

  ' configurable values
  Config.DefaultFileExtension = ".csv"

  ' Excel implemented fixed values
  Config.MaxOfAllowedSheets = 255
  Config.MaxSheetNameLength = 31

  ' dependent values
  Config.MaxDuplicateExtension = " (" & Config.MaxOfAllowedSheets & ")"

  ' set configurable quasi constants
  With Worksheets("Indikator-Files importieren")
    Config.IndicatorMark = .Range("B3")
    Config.DefaultFolderName = .Range("B4")
  End With
  Config.IndicatorRegExpPattern = ".*\..*\.(" & Config.IndicatorMark & "\([^)]*\)).*"
  Config.FileNamePattern = "*.*." & Config.IndicatorMark & "(*)"

End Sub


Sub ImportIndicatorFiles()
'
' Macro main extry to import all .csv files with indicator mark
' (defined above) in file name
'

  InitConfig

  ' dialog for selection of files to process
  With Application.FileDialog(msoFileDialogOpen)
    .Title = "Zu ffnende Indikator-Dateien auswhlen"
    .InitialFileName = Config.DefaultFolderName _
      & Config.FileNamePattern & Config.DefaultFileExtension
    .Show

    ' sorting file names as string array because the
    ' used QuickSort function wasn't ready made for objects
    ReDim FileNames(.SelectedItems.Count) As String
    Dim FileNameIndex As Integer
    For FileNameIndex = 1 To .SelectedItems.Count
      FileNames(FileNameIndex) = .SelectedItems(FileNameIndex)
    Next FileNameIndex
    QuickSort FileNames

    ' call the import operation for every single file
    For FileNameIndex = 1 To .SelectedItems.Count
      ImportOneFile FileNames(FileNameIndex)
    Next FileNameIndex

  End With

End Sub


Private Sub ImportOneFile(FileName As String)
'
' Opens one file .csv file with indicator data
' in a new sheet and adds a header line
'

  ' add new sheet
  With Sheets.Add(After:=Sheets(Sheets.Count))

    ' import CSV file
    With .QueryTables.Add(Connection:= _
      "TEXT;" + FileName, Destination:=Range("A1"))

      .FieldNames = True
      .RowNumbers = False
      .FillAdjacentFormulas = False
      .PreserveFormatting = True
      .RefreshOnFileOpen = False
      .RefreshStyle = xlInsertDeleteCells
      .SavePassword = False
      .SaveData = True
      .AdjustColumnWidth = True
      .RefreshPeriod = 0
      .TextFilePromptOnRefresh = False
      .TextFilePlatform = 850
      .TextFileStartRow = 1
      .TextFileParseType = xlDelimited
      .TextFileTextQualifier = xlTextQualifierDoubleQuote
      .TextFileConsecutiveDelimiter = False
      .TextFileTabDelimiter = False
      .TextFileSemicolonDelimiter = True
      .TextFileCommaDelimiter = False
      .TextFileSpaceDelimiter = False
      .TextFileColumnDataTypes = Array(5, 1, 1, 1, 1, 1)
      .TextFileDecimalSeparator = "."
      .TextFileThousandsSeparator = ","
      .TextFileTrailingMinusNumbers = True
      .Refresh BackgroundQuery:=False
    End With

    ' name the sheet
    .Name = RelevantFileNamePart(FileName)

    ' add a header line
    .Rows("1:1").Select
    Selection.Insert Shift:=xlDown
    .Range("A1").Select
    ActiveCell.FormulaR1C1 = "Time"
    .Range("B1").Select
    ActiveCell.FormulaR1C1 = "Open"
    .Range("C1").Select
    ActiveCell.FormulaR1C1 = "High"
    .Range("D1").Select
    ActiveCell.FormulaR1C1 = "Low"
    .Range("E1").Select
    ActiveCell.FormulaR1C1 = "Close"
    .Range("F1").Select
    ActiveCell.FormulaR1C1 = IndicatorFileNamePart(.Name)
    .Rows("1:1").Select
    Selection.Font.Bold = True

  End With

End Sub


Private Function RelevantFileNamePart(FileName As String) As String
'
' cut the file name to the part for sheet name
'

  ' drop default extension
  If Right(FileName, Len(Config.DefaultFileExtension)) = Config.DefaultFileExtension Then _
    FileName = Left(FileName, Len(FileName) - Len(Config.DefaultFileExtension))

  ' drop directory name
  Dim BackSlashPos
  BackSlashPos = InStrRev(FileName, "\")
  If BackSlashPos > 0 Then FileName = Mid(FileName, BackSlashPos + 1)

  ' cut to the Max length
  FileName = Left(FileName, Config.MaxSheetNameLength - Len(Config.MaxDuplicateExtension))

  'sheet name duplicate check and renaming (if necessary)
  If SheetNameCollision(FileName) Then ' try to rename
    Dim RenameCount As Integer
    Dim TryName As String

    For RenameCount = 2 To Config.MaxOfAllowedSheets
      TryName = FileName & " (" & RenameCount & ")"
      If Not SheetNameCollision(TryName) Then
        FileName = TryName
        Exit For
      End If
    Next RenameCount
  End If

  RelevantFileNamePart = FileName
End Function


Private Function IndicatorFileNamePart(SheetName As String) As String
'
' cut the indicator name with period parameter form the sheet name
'

  Dim RegExp As Object

  Set RegExp = CreateObject("VBScript.RegExp")
  RegExp.IgnoreCase = True
  RegExp.Global = True
  RegExp.Pattern = Config.IndicatorRegExpPattern

  IndicatorFileNamePart = Config.IndicatorMark   ' Default value without periods

  With RegExp.Execute(SheetName)
    If .Count > 0 Then
      If .Item(0).SubMatches.Count > 0 Then
        IndicatorFileNamePart = .Item(0).SubMatches(0)
      End If
    End If
  End With

End Function


Function SheetNameCollision(FileName As String) As Boolean
'
' checks if a sheet with same name already exists
'
  Dim SheetIndex As Integer

  SheetNameCollision = False

  For SheetIndex = 1 To Sheets.Count
    If Sheets(SheetIndex).Name = FileName Then
      SheetNameCollision = True
      Exit Function
    End If
  Next SheetIndex

End Function


' Since VBA 2005 the is a builtin method Array.Sort()
' But this doesn't work with my Excel 6.5 from 2006
' So I used a sort implementation from
' Autor: John Green
' URL: http://de.wikibooks.org/wiki/VBA_in_Excel_-_Grundlagen:_Sortieren

Private Sub QuickSort(ByRef VA_array, Optional V_Low1, Optional V_high1)
    On Error Resume Next
    Dim V_Low2, V_high2, V_loop As Integer
    Dim V_val1, V_val2 As Variant
    If IsMissing(V_Low1) Then
        V_Low1 = LBound(VA_array, 1)
    End If
    If IsMissing(V_high1) Then
        V_high1 = UBound(VA_array, 1)
    End If
    V_Low2 = V_Low1
    V_high2 = V_high1
    V_val1 = VA_array((V_Low1 + V_high1) / 2)
    While (V_Low2 <= V_high2)
        While (VA_array(V_Low2) < V_val1 And _
            V_Low2 < V_high1)
            V_Low2 = V_Low2 + 1
        Wend
        While (VA_array(V_high2) > V_val1 And _
            V_high2 > V_Low1)
            V_high2 = V_high2 - 1
        Wend
        If (V_Low2 <= V_high2) Then
            V_val2 = VA_array(V_Low2)
            VA_array(V_Low2) = VA_array(V_high2)
            VA_array(V_high2) = V_val2
            V_Low2 = V_Low2 + 1
            V_high2 = V_high2 - 1
        End If
    Wend
    If (V_high2 > V_Low1) Then Call _
        QuickSort(VA_array, V_Low1, V_high2)
    If (V_Low2 < V_high1) Then Call _
        QuickSort(VA_array, V_Low2, V_high1)
End Sub
